#include <Gc.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define MAX_TEST 77

extern void (*(test_samples[MAX_TEST]))( TGC* );



/******************************************************************************
*
*  Render
*
*******************************************************************************
*  Draw the example images.
*
*******************************************************************************
*  Usage:
*     Render( TGC* gc );
*
*  Arguments:
*     gc        - the graphics context being drawn on.
*
*  Returns:
*     bool completed - true if the image could be drawn.
*                    - false if it can't be drawn.
*
*******************************************************************************
*/
bool Render_samples( TGC* gc )
{
  INT32 testindex, x, y;
  char  str[100];

  gc->Versions( MAX_TEST + 1 );
  gc->SetLut( BW|GRAYIBM|HUERGB|HUECMY );
  gc->SelectFont( "SYS16" );

  //----------------------------------------------------------------------
  //  If not in Sub-image mode, display instruction on how to get
  //  into the sub-images.
  //----------------------------------------------------------------------
  testindex = gc->GetVersions();
  if( testindex == 0 )
  {
    // Clear the screen
    gc->WipeActive();

    gc->Opaque( TStdColor::red, TStdColor::white );
    gc->DrawString( 20, 50,  "Graphics Examples" );
    gc->DrawString( 20, 80,  "Use the command: ISUB 1; IVER n; IMGU" );
    sprintf( str, "Where n = 1 to %d", MAX_TEST );
    gc->DrawString( 20, 110, str );
  }
  else
  {
    if( testindex > MAX_TEST + 1 ) testindex = MAX_TEST + 1;

    // Execute the selected test
    (*test_samples[testindex-1])(gc);
  }

  return true;
}



//----------------------------------------------------------------------------
// CharWideMax Test Image
//----------------------------------------------------------------------------
void Test_CharWideMax( TGC* gc )
{
  char   c;
  INT32  w, h, x, y,i;
  char   s[] = "QUANTUM DATA";

  //*** Setup look-up-table for colors white and black
  gc->SetLut( BW );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Select the BIG30 font via name
  gc->SelectFont( "BIG30" );

  //*** Set the text color to be white
  gc->Transparent( TStdColor::white );

  //*** Get the text's dimensions
  x = w = gc->CharWideMax();
  y = h = gc->CharHigh();

  //*** Draw the text one char down and one char over from the last
  i = 0;
  while( (c = s[i])  != '\0' )
  {
    gc->DrawChar( x, y, c );

    y += h;
    x += w;
    i++;
  }
}


//----------------------------------------------------------------------------
// 	This is an example image to test the CharHigh() function
// 	this code produces an image that has three lines of text
//  drawn in three rows
//----------------------------------------------------------------------------
void Test_CharHigh( TGC* gc )
{
  char *s[] =
  {
    "1st line",
    "2nd line",
    "3rd line"
  };
  INT32 i, x, y;

  //*** Setup look-up-table for colors white and black
  gc->SetLut( BW );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Select the BIG30 font via index.
  gc->SelectFont( "BIG30" );

  //*** Set the text color to be white
  gc->Transparent( TStdColor::white );

  x = 8;
  y = gc->CharHigh();
  for (i = 0; i <= 2; ++i)
  {
    gc->DrawString(x, y, s[i]);
    y += gc->CharHigh();
  }
}


//----------------------------------------------------------------------------
// 	This is an example image to test the CharWideImage() function
// 	this code produces an image that has a red diagnol line of text
// 	starting from the upper left corner of the screen.  Each character
//  should fit within a series of vertical white lines with no overlap.
//----------------------------------------------------------------------------
void Test_CharWideImage( TGC* gc )
{
  char  c;
  INT32 w, h, x, x1, y, y2, i;
  char s[] = "QUANTUM DATA";

  //*** Setup look-up-table for colors white and black and Red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Select the BIG30 font via name
  gc->SelectFont( "BIG30" );

  //*** Initialize variables
  x  = gc->CharWideMax();
  y  = h = gc->CharHigh();
  y2 = h * strlen( s );
  i  = 0;

  //*** Draw the initial column line
  gc->Set_color1( TStdColor::white );
  gc->DrawLine( x - 1, 0, x - 1, y2 );

  while( (c = s[i])  != '\0' )
  {
    //*** Determine the current characters image width
    w = gc->CharWideImage(c);

    //*** Draw a white column line
    x1 = x + w + 1;
    gc->Set_color1( TStdColor::white );
    gc->DrawLine( x1, 0, x1, y2 );

    //*** Set the text color to be red
    gc->Transparent( TStdColor::red );

    //*** Draw a character
    gc->DrawChar( x, y, c );

    //*** Move to the next character column
    y += h;
    x += w + 2;
    i++;
  }
}



//----------------------------------------------------------------------------
// 	This is an example image to test the GetFirstCh() function
// 	this code produces an image that has 2 lines of white text with all
//  visible ascii chars displayed
//----------------------------------------------------------------------------
void Test_GetFirstCh( TGC* gc )
{
  UINT8 c;
  INT32 x, y;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Select the OPIX9 font via name
  gc->SelectFont( "OPIX9" );

  //*** set the text color to be white
  gc->Transparent( TStdColor::white );

  x = 10;
  y = 100;
  for( c = gc->GetFirstCh(); c <= gc->GetLastCh(); ++c, x += gc->CharWideMax() )
  {
    if( x > 600 )
    {
      x = 10;
      y += 50;
    }
    gc->DrawChar( x, y, c );
  }
}


//----------------------------------------------------------------------------
// 	This is an example image to test the GetLastCh() function
// 	This code produces an image that has 2 lines of white text with
// 	all visible ascii chars displayed
//----------------------------------------------------------------------------
void Test_GetLastCh( TGC* gc )
{
  UINT8 c;
  INT32 x, y;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Select the OPIX9 font via name
  gc->SelectFont( "OPIX9" );

  //*** Set the text color to be white
  gc->Transparent( TStdColor::white );

  x = 10;
  y = 100;
  for( c = gc->GetFirstCh(); c <= gc->GetLastCh(); ++c, x += gc->CharWideMax() )
  {
    if (x > 600)
     {
      x = 10;
      y += 50;
     }
     gc->DrawChar( x, y, c );
  }
}


//----------------------------------------------------------------------------
// 	This is an example image to test the CurtainFillRect() function
// 	This code produces an image that has a rectangle of blue horizontal
//  lines and a rectangle of vertical red, green, blue lines.
//----------------------------------------------------------------------------
void Test_CurtainFillRect( TGC* gc )
{
  INT32 x, y;

  //*** Setup look-up-table for colors white and black and Red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the text color to be blue
  gc->Transparent( TStdColor::blue );

  //*** Draw some vertical lines along the right edge of the screen
  for( x = 0, y = 0; y < 100; y += 5 )
  {
    gc->DrawLine( x, y, x, y+2 );
  }

  //*** Curtain Fill the vertical lines, filling left to right
  gc->CurtainFillRect( 100, 100, 0, 0, 1 );

  //*** Draw some horizontal lines along the top edge
  for( x = 150, y = 0; x < 255; x += 15 )
  {
    gc->Set_color1( TStdColor::red );
    gc->DrawLine( x, y, x+5, y );
    gc->Set_color1( TStdColor::green );
    gc->DrawLine( x+6, y, x+10, y );
    gc->Set_color1( TStdColor::blue );
    gc->DrawLine( x+11, y, x+14, y );
  }

  //*** Curtain fill the horizontal lines, top to bottom
  gc->CurtainFillRect( 106, 200, 150, 0, 0 );
}


//----------------------------------------------------------------------------
//	This is an example image to test the DrawChar() function
//	This code produces an image that has white letters arranged
//	in a circle about the middle of the screen
//----------------------------------------------------------------------------
void Test_DrawChar( TGC* gc )
{
  char  c;
  INT32 x, y;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
   gc->BlackFillVuport();

  //*** Select the OPIX9 font via name
  gc->SelectFont( "OPIX9" );

  //*** Set the text color to be white
  gc->Transparent( TStdColor::white );

  x = 0;
  y = -170 << 16;

  //*** Draw the letters 'A' through 'Z'
  for( c = 'A'; c < 'Z' ; ++c )
  {
    //*** draw the characters on screen
    gc->DrawChar( (x >> 16) + 304, (y >> 16) + 244, c );
    x += y >> 3;
    y -= x >> 3;

    //*** draw the characters on screen
    gc->DrawChar( (x >> 16) + 304, (y >> 16) + 244, c - 'A' + 'a' );
    x += y >> 3;
    y -= x >> 3;
  }
}


//----------------------------------------------------------------------------
// 	This is an example image to test the DrawLine() function
//	This code produces an image that has many lines drawn
//  around a circle cenetred in the middle of the screen.
//----------------------------------------------------------------------------
void Test_DrawLine( TGC* gc )
{
  INT32 i, x1 = -40, y1 = 80, x2 = 80, y2 = 280;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut(BW|HUERGB);

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Draw 202 lines in different orientations and colors
  for( i = 202; i > 0; --i )
  {
    // Alternate the line colors (rgb)
    switch( i % 3 )
    {
      case 0:
        gc->Set_color1( TStdColor::red );
        break;

      case 1:
        gc->Set_color1( TStdColor::green );
        break;

      case 2:
        gc->Set_color1( TStdColor::blue );
    }

    // Draw a line
    gc->DrawLine( x1 + 305, y1 + 222, x2 + 305, y2 + 222 );

    // Move to the next line
    x1 += y1 >> 5;
    y1 -= x1 >> 5;
    x2 += y2 >> 5;
    y2 -= x2 >> 5;
  }
}


//----------------------------------------------------------------------------
//	 	This is an example image to test the DrawOval() function.
//   	This code produces an image that has white ovals of various sizes
//----------------------------------------------------------------------------
void Test_DrawOval( TGC* gc )
{
  INT32 w, h, x, y;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the color to be white
  gc->Transparent( TStdColor::white );

  //*** Draw ellipses of various sizes
  for( w = 0, x = 4; w < 33; ++w, x += w + 3 )
    for( h = 0, y = 4; h < 28; ++h, y +=h + 3 )
      gc->DrawOval( w, h, x, y );
}


//----------------------------------------------------------------------------
// 	This is an example image to test the DrawPoint() function.
// 	This code produces an image that has figure 8 lines in green
//----------------------------------------------------------------------------
void Test_DrawPoint( TGC* gc )
{
  INT32 i, x, y, xy, yx;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the color to be green
  gc->Transparent( TStdColor::green );

  x = xy = 0;
  y = yx = 200;
  for( i = 1200; i > 0; --i )
  {
    gc->DrawPoint( x + 320, y + 240);
    x  += yx >> 4;
    yx -= x  >> 4;
    y  += xy >> 5;
    xy -= y  >> 5;
  }
}


//----------------------------------------------------------------------------
// 	This is an example image to test the DrawPolyLine() function.
// 	This code produces an image that has a white cube
//----------------------------------------------------------------------------
void Test_DrawPolyLine( TGC* gc )
{
  POINT ptlist[7];
  POINT linelist[9];

  ptlist[0] = 0x017C00C8;
  ptlist[1] = 0x01E000C8;
  ptlist[2] = 0x01E0012C;
  ptlist[3] = 0x017C012C;
  ptlist[4] = 0x0154010E;
  ptlist[5] = 0x015400AA;
  ptlist[6] = 0x01BB00AA;


  linelist[0] = 0x00000001;
  linelist[1] = 0x00010002;
  linelist[2] = 0x00020003;
  linelist[3] = 0x00030004;
  linelist[4] = 0x00040005;
  linelist[5] = 0x00050006;
  linelist[6] = 0x00060001;
  linelist[7] = 0x00030000;
  linelist[8] = 0x00050000;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the color to be white
  gc->Transparent( TStdColor::white );

  //*** Draw a cube
  gc->DrawPolyLine( 9, linelist, ptlist );
}


//----------------------------------------------------------------------------
// 	This is an example image to test the DrawRect() function.
// 	This code produces an image that has white rectangles drawn
// 	one inside another
//----------------------------------------------------------------------------
void Test_DrawRect( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the color to be white
  gc->Transparent( TStdColor::white );

  gc->DrawRect( 440, 280, 100, 100 );
  gc->DrawRect( 420, 30,  110, 110 );
  gc->DrawRect( 220, 220, 110, 150 );
  gc->DrawRect( 190, 150, 340, 150 );
  gc->DrawRect( 190, 60,  340, 310 );
}


//----------------------------------------------------------------------------
// 	This is an example image to test the DrawTriangle() function.
// 	This code produces an image that has 3 triangles drawn; a blue
// 	and green inside of a white one
//----------------------------------------------------------------------------
void Test_DrawTriangle( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the color to be white
  gc->Transparent( TStdColor::white );

  gc->DrawTriangle( 100, 100, 500, 100, 320, 400 );

  //*** Set the color to be blue
  gc->Transparent( TStdColor::blue );
  gc->DrawTriangle( 160, 160, 420, 160, 320, 320 );

  //*** Set the color to be green
  gc->Transparent( TStdColor::green );
  gc->DrawTriangle( 240, 220, 340, 220, 320, 260 );
}


//----------------------------------------------------------------------------
// 	This is an example image to test the DrawCenterMark() function.
// 	This code produces an image that has a white crosshair in the center
// 	of the screen
//----------------------------------------------------------------------------
void Test_DrawCenterMark( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the color to be white
  gc->Transparent( TStdColor::white );

  //*** Draw a cross in the center of the screen
  gc->DrawCenterMark();
}


//----------------------------------------------------------------------------
// 	This is an example image to test the DrawLimits() function.
// 	Tthis code produces an image that has white  corner markers,
// 	centermarker, and markers at the center of the screen sides.
//----------------------------------------------------------------------------
void Test_DrawLimits( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the color to be white
  gc->Transparent( TStdColor::white );

  //*** Draw a cross with the other limits on the screen
  gc->DrawLimits();
}



//----------------------------------------------------------------------------
// 	This is an example image to test the DrawCross() function.
// 	This code produces an image that has a white cross on the screen
// 	dividing it into 4 parts
//----------------------------------------------------------------------------
void Test_DrawCross( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the color to be white
  gc->Transparent( TStdColor::white );

  //*** Draw a cross in the center of the screen
  gc->DrawCross();
}


//----------------------------------------------------------------------------
// This is an example image to test the DrawTBox() function.
// This code produces an image that has two information boxes.
//----------------------------------------------------------------------------
void Test_DrawTBox( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Set_color1( TStdColor::green );
  gc->Set_color0( TStdColor::red   );

  //*** Show the information about the current image displayed in a box
  gc->DrawTextBox( 320, 200 );
  gc->DrawTextBox( 320, 280, TStdColor::white );
}


//----------------------------------------------------------------------------
// This is an example image to test the DrawFormat() function.
// This code produces an image that has two FORMAT information boxes.
//----------------------------------------------------------------------------
void Test_DrawFormat( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Set_color1( TStdColor::green );
  gc->Set_color0( TStdColor::red   );

  gc->SelectFont( "OPIX9" );

  gc->DrawFormat( 320, 200 );
  gc->DrawFormat( 320, 280, TStdColor::white );
}


//----------------------------------------------------------------------------
// 	This is an example image to test the DrawGrid() function.
// 	This code produces an image that has a dotted hatch grid 16*12
//----------------------------------------------------------------------------
void Test_DrawGrid( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Transparent( TStdColor::green );
  gc->DrawGrid( 16, 12, 31 );
}


//----------------------------------------------------------------------------
// 	This is an example image to test the DrawGrill() function.
// 	This code produces an image that has vertical and horizontal
// 	green bars
//----------------------------------------------------------------------------
void Test_DrawGrill( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Transparent( TStdColor::green );
  gc->DrawGrill( 600, 300, 30, 30, 20, 15, 2 );
}


//----------------------------------------------------------------------------
// 	This is an example image to test the DrawString() function.
// 	This code produces an image that has hello world drawn 50 times in
//	a circular pattern
//----------------------------------------------------------------------------
void Test_DrawString( TGC* gc )
{
  INT32  i;
  INT32  x, y;
  char   *s;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Transparent( TStdColor::white );

  gc->SelectFont( "OPIX9" );
  s = "Hello world.";
  x = 0;
  y = 200 << 16;

  //*** Write 'Hello world' to the screen 50 times
  for( i = 50; i> 0; --i )
  {
    gc->DrawString( (x >> 16) + 272, (y >> 16) + 244, s );
    x += y >> 3;
    y -= x >> 3;
  }
}


//----------------------------------------------------------------------------
//	This is an example image to test the DrawText() function.
// 	This code produces an image that has hello world drawn 50 times in
// 	a circular pattern
//----------------------------------------------------------------------------
void Test_DrawText( TGC* gc )
{
  INT32  i;
  INT32  x, y;
  char   *s;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Transparent( TStdColor::white );

  gc->SelectFont( "OPIX9" );
  s = "Hello world.";
  x = 0;
  y = 200 << 16;

  //*** Write 'Hello world' to the screen 50 times
  for (i = 50; i> 0; --i)
  {
    gc->DrawText( (x >> 16) + 272, (y >> 16) + 244, s );
    x += y >> 3;
    y -= x >> 3;
  }
}


//----------------------------------------------------------------------------
//	This is an example image to test the GetWidth() function.
//	This code produces an image that has white cross lines with
//  red text centered between the two left and right sides of
//  the screen
//----------------------------------------------------------------------------
void Test_GetWidth( TGC* gc )
{
  char *s[] =
  {
    "The GetWidth() function",
    "is easily used",
    "on your screen."
  };
  INT32 x, y, i;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the color to be white
  gc->Transparent( TStdColor::white );

  //*** Crosshairs
  gc->DrawLine( 320,   0,   320, 2*240 );
  gc->DrawLine( 0,   240, 2*320,   240 );

  y = 240 - (4 * gc->CharHigh()/2) + gc->GetAscent();

  //*** Set the color to be red
  gc->Transparent( TStdColor::red );

  gc->SelectFont( "BIG30" );
  for( i = 0; i < 3; ++i )
  {
    x = 320 - gc->GetWidth(s[i])/2;
    gc->DrawString( x, y, s[i] );
    y += gc->CharHigh();
  }
}


//----------------------------------------------------------------------------
// 	This is an example image to test the CharFillRect() function.
// 	This code produces an image that has a rectangular region filled
// 	with the red C's and another region filled with the '#' character.
//----------------------------------------------------------------------------
void Test_CharFillRect( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the color to be red
  gc->Transparent( TStdColor::red );

  gc->SelectFont( "BIG30" );

  //*** Fill a rectangle with the character C
  gc->CharFillRect( 100, 80, 10, 20, 2, 1, 0, 0, 'C' );

  //*** Fill another rectangle with '#'
  gc->SelectFont( "sys16" );
  gc->FillRect( 200, 200, 200, 200 );
  gc->Opaque( TStdColor::white, TStdColor::black );
  gc->CharFillRect( 200, 200, 200, 200, 5, 5, 5, 5, '#' );
}



//----------------------------------------------------------------------------
//  This is an example image to test the FillOval() function.
//  This code produces an image that has green filled elipses
//  in different sizes
//----------------------------------------------------------------------------
void Test_FillOval( TGC* gc )
{
  INT32 w, h, x, y;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the color to be green
  gc->TranspOFF();
  gc->Set_color1( TStdColor::green );

  //*** Fill ellipses of various sizes
  for( w = 0, x = 4; w < 33; ++w, x += w + 3 )
    for( h = 0, y = 4; h < 28; ++h, y += h + 3 )
      gc->FillOval( w, h, x, y );
}


//----------------------------------------------------------------------------
//  This is an example image to test the FillRect() function.
//  This code produces an image that has 5 filled rectangles
//  of various sizes and colors
//----------------------------------------------------------------------------
void Test_FillRect( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the color to be white
  gc->TranspOFF();
  gc->Set_color1( TStdColor::white );

  //*** Draw a filled rectangle
  gc->FillRect( 440, 280, 100, 100 );

  gc->Set_color1( TStdColor::red );
  gc->FillRect( 420, 30, 110, 110 );

  gc->Set_color1( TStdColor::green );
  gc->FillRect( 220, 220, 110, 150 );

  gc->Set_color1( TStdColor::blue );
  gc->FillRect( 190, 150, 340, 150 );

  gc->Set_color1( TStdColor::black );
  gc->FillRect( 190, 60, 340, 310 );
}


//----------------------------------------------------------------------------
//  This is an example image to test the FillTriangle() function.
//  This code produces an image that has 3 multicolored, filled triangles
//----------------------------------------------------------------------------
void Test_FillTriangle( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();
  gc->TranspOFF();

  //*** Set the color to be white
  gc->Set_color1( TStdColor::white );
  gc->FillTriangle( 100, 100, 500, 100, 320, 400 );

  gc->Set_color1( TStdColor::blue );
  gc->FillTriangle( 160, 160, 420, 160, 320, 320 );

  gc->Set_color1( TStdColor::green );
  gc->FillTriangle( 240, 220, 340, 220, 320, 260 );
}


//----------------------------------------------------------------------------
//  This is an example image to test the FillPolygon() function.
//  This code produces an image that has a star-shaped filled polygon
//----------------------------------------------------------------------------
void Test_FillPolygon( TGC* gc )
{
  INT32 n = 5;
  POINT points[5];

  points[0] = 0x003200F0;
  points[1] = 0x026C0082;
  points[2] = 0x00B401C2;
  points[3] = 0x01400032;
  points[4] = 0x01A401A4;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();
  gc->TranspOFF();

  //*** Set the color to be green
  gc->Set_color1( TStdColor::green );
  gc->FillPolygon( n, points );
}


//----------------------------------------------------------------------------
//  This is an example image to test the GetAscent() function.
//  This code produces an image that has white text in a red box
//  at upper left corner of the screen
//----------------------------------------------------------------------------
void Test_GetAscent( TGC* gc )
{
  char *s = "Hello world.";
  INT32 w, h, x, y;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the font
  gc->SelectFont( "OPIX9" );

  //*** Draw text flush with top of screen
  x = y = 0;

  h = gc->GetAscent();
  w = gc->GetWidth(s);

  gc->Transparent( TStdColor::red );
  gc->FillRect( w, h, x, y );

  gc->Transparent( TStdColor::white );
  gc->DrawString( x, y+h, s );
}


//----------------------------------------------------------------------------
//  This is an example image to test the GetDescent() function.
//  This code produces an image that has white text with a red
// 	line drawn just below descenders
//----------------------------------------------------------------------------
void Test_GetDescent( TGC* gc )
{
  char  *s = "jumping jimminy";
  INT32 w, h, x, y;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->SelectFont( "OPIX9" );
  gc->Transparent( TStdColor::white );

  //*** Draw line just below descenders
  x = 0;
  y = gc->GetAscent();
  w = gc->GetWidth(s);
  h = gc->GetDescent();
  gc->DrawString( x, y, s );
  gc->Transparent( TStdColor::red );
  gc->FillRect( w, 1, x, y+h );
}


//----------------------------------------------------------------------------
//  This is an example image to test the GetLeading() function.
//  This code produces an image that has white text written
//  above white lines row by row with increasing space from
//  the left side of the screen
//----------------------------------------------------------------------------
void Test_GetLeading( TGC* gc )
{
  char  s[] = "Quantum Data...";
  INT32 x, y, dx;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the active color
  gc->Transparent( TStdColor::white );

  //*** Set the font
  gc->SelectFont( "OPIX9" );

  dx = (640 - gc->GetWidth(s)) / (480 / gc->CharHigh());
  for( x = 0, y = gc->GetAscent(); y < 480; x += dx, y += gc->CharHigh() )
  {
    gc->DrawString( x, y, s );
    gc->FillRect( 640, 1, 0, y + gc->GetDescent() + gc->GetLeading() /2 );
  }
}


//----------------------------------------------------------------------------
//  This is an example image to test the MoveRect() function.
//  This code produces an image at the upper left corner and moves
//  part of it to the lower right portion of the screen
//----------------------------------------------------------------------------
void Test_MoveRect( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Opaque( TStdColor::blue, TStdColor::red );

  gc->SelectPatn( "CHECKER8" );
  gc->PatnFillRect( 100, 100, 0, 0 );
  gc->MoveRect( 90, 90, 5, 5, 300, 300 );

  gc->Opaque( TStdColor::white, TStdColor::black );
  gc->DrawRect( 92, 92, 4, 4 );
  gc->DrawRect( 92, 92, 299, 299 );
}


//----------------------------------------------------------------------------
//  This is an example image to test the Opaque() function.
//  This code produces an image that has yellow text with cyan
//  background
//----------------------------------------------------------------------------
void Test_Opaque( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUECMY );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the Font
  gc->SelectFont( "BIG30" );

  if( TStdColor::yellow && TStdColor::cyan )
  {
    gc->Opaque( TStdColor::yellow, TStdColor::cyan );
    gc->DrawString( 100, 100, "Yellow text with Cyan background" );
  }
}



//----------------------------------------------------------------------------
//  This is an example image to test the PatnFillRect() function.
//  This code produces an image that has 5 pettern filled rects
//----------------------------------------------------------------------------
void Test_PatnFillRect( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Opaque( TStdColor::red, TStdColor::green );
  gc->SelectPatn( "CHECKER8" );
  gc->PatnFillRect( 440, 280, 100, 100 );

  gc->Opaque( TStdColor::white, TStdColor::blue );
  gc->SelectPatn( "MEME" );
  gc->PatnFillRect( 420,  30, 110, 110 );
  gc->PatnFillRect( 220, 220, 110, 150 );
  gc->PatnFillRect( 190, 150, 340, 150 );
  gc->PatnFillRect( 190,  60, 340, 310 );
}


//----------------------------------------------------------------------------
//  This is an example image to test the PatnFillTriangle() function.
//  This code produces an image that has 3 pattern filled triangles
//----------------------------------------------------------------------------
void Test_PatnFillTriangle( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Opaque( TStdColor::white, TStdColor::blue );
  gc->SelectPatn(  "CHECKER8" );
  gc->PatnFillTriangle( 100, 100, 500, 100, 320, 400 );

  gc->Opaque( TStdColor::green, TStdColor::red );
  gc->SelectPatn( "BARS_H2" );
  gc->PatnFillTriangle( 160, 160, 420, 160, 320, 320 );

  gc->Opaque( TStdColor::black, TStdColor::white );
  gc->SelectPatn( "BARS_V2" );
  gc->PatnFillTriangle( 240, 220, 340, 220, 320, 260 );
}



//----------------------------------------------------------------------------
//  This is an example image to test the PatnFillOval() function.
//  This code produces an image that has multiple pattern filled
//  ellipses of different sizes
//----------------------------------------------------------------------------
void Test_PatnFillOval( TGC* gc )
{
  INT32  w, h, x, y, patn;
  UINT32 hue;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();
  gc->TranspOFF();

  //*** Pattern fill ellipses of various sizes
  x = y = hue = 0;
  patn = 1;
  for( w = 640, h = 480; w >= 192; w -= 32, h -= 24 )
  {
    gc->Set_color0( hue += 0x01010101 );
    gc->Set_color1( ~hue );
    gc->SelectPatn( patn++ );
    gc->PatnFillOval( w, h, x, y );
  }
}


//----------------------------------------------------------------------------
//  This is an example image to test the PatnFrameOval() function
//----------------------------------------------------------------------------
void Test_PatnFrameOval( TGC* gc )
{
  INT32 i;

  //*** Setup look-up-table
  gc->SetLut( GRAYACE | HUERGB | HUECMY | HUEPEN );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Select a pattern by name
  gc->SelectPatn( "checker1" );

  //*** Draw some ovals
  gc->Set_color0( TStdColor::black );
  for( i = 0; i <= 10; i++ )
  {
    gc->Set_color1( gc->RepPixel( i+1 ) );
    gc->PatnFrameOval( 640 - i*60, 20 + i*46, i*30, 230 - i*23, 8, 8 );
  }
}


//----------------------------------------------------------------------------
//  This is an example image to test the FrameOval() function.
//----------------------------------------------------------------------------
void Test_FrameOval( TGC* gc )
{
  INT32 i;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();
  gc->TranspOFF();

  for( i = 0; i <= 10; i++ )
  {
    gc->Set_color1( gc->RepPixel( i+1 ) );
    gc->FrameOval( 640 - i*60, 20 + i*46, i*30, 230 - i*23, 8, 8 );
  }
}


//----------------------------------------------------------------------------
//  This is an example image to test the FrameRect() function.
//  This code produces an image that has a solid red framed rectangle
//----------------------------------------------------------------------------
void Test_FrameRect( TGC* gc )
{
  INT32 w, h, x, y, dx, dy;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();
  gc->TranspOFF();
  gc->Set_color1( TStdColor::red );

  w  = 480;
  h  = 360;
  x  =  80;
  y  =  60;
  dx =  40;
  dy =  30;

  gc->FrameRect( w, h, x, y, dx, dy );
}

//----------------------------------------------------------------------------
//  This is an example image to test the PatnFrameRect() function.
//  This code produces an image that has a red and green pattern
//  framed rectangle
//----------------------------------------------------------------------------
void Test_PatnFrameRect( TGC* gc )
{
  INT32 w, h, x, y, dx, dy;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the active color
  gc->Opaque( TStdColor::red, TStdColor::green );

  w  = 480;
  h  = 360;
  x  =  80;
  y  =  60;
  dx =  40;
  dy =  30;

  gc->SelectPatn( "CHECKER8" );
  gc->PatnFrameRect( w, h, x, y, dx, dy );
}



//----------------------------------------------------------------------------
// This is an example image to test the SelectColor() function.
// This code produces an image that has vertical bars in different shades
// of red.
//----------------------------------------------------------------------------
void Test_SelectColor( TGC* gc )
{
  INT32 w, x, i;
  INT32 r, g, b;
  INT32 rr, gg, bb;

  //*** Setup look-up-table for colors white and black and colors red, green
  //*** and blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen with black.
  gc->BlackFillVuport();

  w = gc->width / 256;
  x = (gc->width - (w*256))/2;
  g = b = 0;
  for( i = 0; i <= 255; i++ )
  {
    r = i;
    gc->Transparent( gc->RepPixel(i) );
    gc->SelectColor( i );
    rr = gc->MapLevel(r);
    gg = gc->MapLevel(g);
    bb = gc->MapLevel(b);
    gc->SetLevelRGB( rr, gg, bb );
    gc->FillRect( w, gc->height, x, 0 );
    x = x + w;
  }
}

//----------------------------------------------------------------------------
// This is an example image to test the SetColorNamed() function.
// This code produces an image that has a red rectangle
//----------------------------------------------------------------------------
void Test_SetColorNamed( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set up the color red at slot 2 and pick it up
  gc->Transparent( gc->SetColorNamed( "red" ) );

  gc->DrawRect( 100, 100, 100, 100 );
}


//----------------------------------------------------------------------------
//  This is an example image to test the SelectFont() function.
//  This code produces an image that has Hello world printed in
//  green using font at index 1
//----------------------------------------------------------------------------
void Test_SelectFont( TGC* gc )
{
  INT32  x, y;
  char   s[] = "Hello world";

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Transparent( TStdColor::green );

  //*** Select font by index
  gc->SelectFont( 1 );

  x = 320 - gc->GetWidth(s)/2;
  y = gc->CharHigh();
  gc->DrawString( x, y, s );
}


//----------------------------------------------------------------------------
//  This is an example image to test the SelectFontNamed() function.
//  This code produces an image that has Hello world printed in
//  blue using font BIG30
//----------------------------------------------------------------------------
void Test_SelectFontNamed( TGC* gc )
{
  INT32  x, y;
  char   s[] = "Hello world";

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->SelectFontNamed( "BIG30" );

  gc->Transparent( TStdColor::blue );
  x = 320 - gc->GetWidth( s )/2;
  y = gc->CharHigh();
  gc->DrawString( x, y, s );
}


//----------------------------------------------------------------------------
//  This is an example image to test the SelectPatn() function.
//  This code produces an image that has a blue & green checker
//  board
//----------------------------------------------------------------------------
void Test_SelectPatn( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Opaque( TStdColor::green, TStdColor::blue );

  gc->SelectPatn( "CHECKER8" );
  gc->PatnFillRect( 160, 160, 100, 100 );
}


//----------------------------------------------------------------------------
//  This is an example image to test the SelectPatn() function.
//  This code produces an image that has a blue striped box with
//  a 26 in green and blue backgroung text below
//----------------------------------------------------------------------------
void Test_SelectPatnNamed( TGC* gc )
{
  char s[32];

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->SelectFont( "OPIX9" );

  gc->Opaque( TStdColor::green, TStdColor::blue );
  sprintf( s, "%d", gc->SelectPatn( "bars_H2" ) );

  gc->DrawString( 180, 280, s );
  gc->PatnFillRect( 160, 160, 100, 100 );
}


//----------------------------------------------------------------------------
// This is an example image to test the PokeColor() and the
// PokeLevelRGB() functions.
// This code produces an image that has Rectangles filled with
// various shades of red and blue
//----------------------------------------------------------------------------
void Test_PokeColor( TGC* gc )
{
  INT32 w, x, i;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the active color
  gc->Transparent( TStdColor::red );

  w = gc->width / 5;
  x = (gc->width - (w*5)) / 2;
  for(i=0; i<=4; i++)
  {
    gc->Transparent( gc->RepPixel( i ) );
    gc->PokeColor( i );
    gc->PokeLevelRGB( i*250, 0, i*250 );   // mix of red and blue colors
    gc->FillRect( w, gc->height, x, 0 );
    x = x + w;
  }
}


//----------------------------------------------------------------------------
// This is an example image to test the PokeBlackColors() function
// this code produces an image that has Rectangles filled with
// various shades of red and blue
//----------------------------------------------------------------------------
void Test_PokeBlackColors( TGC* gc )
{
  INT32 w, x, i;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the active color
  gc->Transparent(TStdColor::red);

  w = gc->width / 5;
  x = (gc->width - (w*5)) / 2;
  for(i=0; i<=4; i++)
  {
    gc->Transparent( gc->RepPixel(i) );
    gc->PokeColor( i );
    gc->PokeLevelRGB( i*250, 0, i*250 );   // mix of red and blue colors
    gc->FillRect( w, gc->height, x, 0 );
    x = x + w;
  }

  //*** Set the intensities of red, green, and blue primaries of the currently
  //*** selected color code in an analog look-up-table to zero level.
  gc->PokeBlackColors();
}


//----------------------------------------------------------------------------
// This is an example image to test the SetLevelGray() function.
// This code produces an image that has 5 gray rectangles
//----------------------------------------------------------------------------
void Test_SetLevelGray( TGC* gc )
{
  INT32 w, x, i;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | GRAYDEC | GRAYEND | GRAYCON );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the active color
  gc->Transparent( TStdColor::red );

  w = gc->width/5;
  x = (gc->width - (w*5))/2;
  for( i = 0; i <= 4; i++ )
  {
    gc->Transparent( gc->RepPixel( i ) );
    gc->SelectColor( i );
    gc->SetLevelGray( i*20 );
    gc->FillRect( w, gc->height, x, 0 );
    x = x + w;
  }
}

//----------------------------------------------------------------------------
// This is an example image to test the SetLevelGrayPercent() function.
// This code produces an image that has 5 gray rectangles
//----------------------------------------------------------------------------
void Test_SetLevelGrayPercent( TGC* gc )
{
  INT32 w, x, i;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | GRAYDEC | GRAYEND | GRAYCON );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the active color
  gc->Transparent( TStdColor::red );

  w = gc->width / 5;
  x = (gc->width - (w*5)) / 2;
  for( i = 0; i <= 4; i++ )
  {
    gc->Transparent( gc->RepPixel( i ) );
    gc->SelectColor( i );
    gc->SetLevelGrayPercent( i * 25 );
    gc->FillRect( w, gc->height, x, 0 );
    x = x + w;
  }
}

//----------------------------------------------------------------------------
// This is an example image to test the SetLevelRGB() function.
//----------------------------------------------------------------------------
void Test_SetLevelRGB( TGC* gc )
{
  INT32 w, x, i;
  INT32 r, g, b;
  INT32 rr, gg, bb;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the active color
  gc->Transparent( TStdColor::blue );

  w = gc->width / 256;
  x = (gc->width - (w*256)) / 2;
  r = g = 0;
  for( i = 0; i <= 255; i++ )
  {
    b = i;
    gc->Transparent( gc->RepPixel( i ) );
    gc->SelectColor( i );
    rr = gc->MapLevel(r);
    gg = gc->MapLevel(g);
    bb = gc->MapLevel(b);
    gc->SetLevelRGB( rr, gg, bb );
    gc->FillRect( w, gc->height, x, 0 );
    x = x + w;
  }
}



//----------------------------------------------------------------------------
//  This is an example image to test the TransformDX() function.
//  Rhis code produces an image that has a point at top side and
//  halfway right + 10 machine units
//----------------------------------------------------------------------------
void Test_TransformDX( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the active color
  gc->Transparent( TStdColor::white );

  gc->ResetMatrix();

  //*** Draw a point at top middle + 10 machine units
  gc->DrawPoint( (gc->TransformDX( gc->normalwidth/2 ) + 10), 0 );
}


//----------------------------------------------------------------------------
//  This is an example image to test the TransformDY() function.
//  This code produces an image that has a point at left side and
//  halfway down + 10 machine units
//----------------------------------------------------------------------------
void Test_TransformDY( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the active color
  gc->Transparent( TStdColor::white );

  gc->ResetMatrix();

  //*** Draw a point at left side and halfway down + 10 machine units
  gc->DrawPoint( 0, (gc->TransformDY( gc->normalheight/2 ) + 10) );
}


//----------------------------------------------------------------------------
//  This is an example image to test the SquareDX() function.
//  This code produces an image that has a red circle of radius
//  equal to the height of the screen
//----------------------------------------------------------------------------
void Test_SquareDX( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the active color
  gc->Transparent( TStdColor::red );

  //*** Draw a circle of radius equals height of the screen
  gc->DrawOval( gc->SquareDX( gc->height ), gc->height - 1,
                gc->centerleft - (gc->SquareDX( gc->height )/2), gc->top );
}


//----------------------------------------------------------------------------
//  This is an example image to test the SquareDY() function.
//  This code produces an image that has a red circle of radius
//  equal to the half width at center of the screen
//----------------------------------------------------------------------------
void Test_SquareDY( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the active color
  gc->Transparent( TStdColor::red );

  //*** Draw a circle of radius equals height of the screen
  gc->DrawOval( gc->width/2, gc->SquareDY( gc->width/2 ),
                gc->centerleft - (gc->width/4),
                gc->centertop - gc->SquareDY( gc->width/4 ) );
}


//----------------------------------------------------------------------------
//  This is an example image to test the TransformX() function.
//  This code produces an image that has 2 red points at top of
//  screen in current normalized axis coordinates
//----------------------------------------------------------------------------
void Test_TransformX( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the active color
  gc->Transparent( TStdColor::red );

  gc->ResetMatrix();

  //*** Draw a point at top of screen and the x coordinate in current
  //*** normalized axis coordinates
  gc->DrawPoint( gc->TransformX(0.75) ,0);

  //*** Draw a point at top of screen and where x=0.0 in current normalized
  //*** axis coordinates.
  gc->DrawPoint( gc->TransformX(0.0) , 0);
}


//----------------------------------------------------------------------------
//  This is an example image to test the TransformY() function.
//  This code produces an image that has 2 white points 1 at top
//  left of screen and the other at left of screen and the y
//  coordinate in current normalized axis coordinates
//----------------------------------------------------------------------------
void Test_TransformY( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the active color
  gc->Transparent( TStdColor::white );

  gc->ResetMatrix();

  //*** Draw a point at left of screen and the y coordinate in current
  //*** normalized axis coordinates
  gc->DrawPoint( 0, gc->TransformY(0.75) );

  //*** Draw a point at top of screen and where y=0.0 in current normalized
  //*** axis coordinates
  gc->DrawPoint( 0, gc->TransformY(0.0) );
}


//----------------------------------------------------------------------------
//  This is an example image to test the Transparent() function.
//  This code produces an image with  "Black" text on a cyan rectangle
//  in black background
//----------------------------------------------------------------------------
void Test_Transparent( TGC* gc )
{
  //*** set up 50% intensity colors and black 	and white
  gc->SetLut( BW | HUEMID );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set up 50% intensity colors and black 	and white
  if( TStdColor::cyan50 )
  {
    gc->Transparent( TStdColor::cyan50 );

    // Fill a rectangle in 50% intensity cyan
    gc->FillRect( 200, 200, 300, 300 );
  }

  gc->Transparent( TStdColor::black );

  gc->SelectFont( "BIG30" );

  //*** Draw "Black" on cyan rectangle in black background is transparent so
  //*** border of characters unaltered.
  gc->DrawText( 350, 350, "Black" );
}


//----------------------------------------------------------------------------
//  This is an example image to test the GroundFillVuport() function.
//  This code produces an image with a red background then fill it
//  with black
//----------------------------------------------------------------------------
void Test_GroundFillVuport( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Fill the screen with red
  gc->ColorFillVuport( TStdColor::red );

  //*** Fill the screen with black (the background)
  gc->GroundFillVuport();
}


//----------------------------------------------------------------------------
//  This is an example image to test the ColorFillVuport() function.
//  This code produces an image with a red background
//----------------------------------------------------------------------------
void Test_ColorFillVuport( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Fill the screen with red
  gc->ColorFillVuport( TStdColor::red );
}


//----------------------------------------------------------------------------
//  This is an example image to test the BlackFillVuport() function.
//  This code produces an image with a black background
//----------------------------------------------------------------------------
void Test_BlackFillVuport( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Fill the screen with black
  gc->BlackFillVuport( );
}


//----------------------------------------------------------------------------
//  This is an example image to test the Boxes() function.
// 	This code produces an image that has a dotted hatch grid
//  24*32
//----------------------------------------------------------------------------
void Test_Boxes( TGC* gc )
{
  INT32 xbox, ybox;

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->Boxes( &xbox, &ybox, 2, -1 );
  gc->Transparent( TStdColor::green );
  gc->DrawGrid( (INT32)xbox, (INT32)ybox, 31 );
}



//----------------------------------------------------------------------------
//  This is an example image to test the TranspOFF and TranspON functions.
//  This code produces an image that has several overlapping rects
//  and pattern filled ovals
//----------------------------------------------------------------------------
void Test_TranspOFF( TGC* gc )
{
  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  gc->SelectPatn( "CHECKER8" );
  gc->Opaque( TStdColor::green, TStdColor::black );
  gc->PatnFrameOval( 150, 150, 45, 165, 35, 35 );

  //*** Construct 2 copies of destination
  gc->Transparent( TStdColor::red );
  gc->FillRect( 190, 190, 225, 145 );
  gc->Transparent(TStdColor::blue);
  gc->FillRect( 30, 190, 305, 145 );
  gc->FillRect( 190, 30, 225, 225 );
  gc->MoveRect( 200, 200, 220, 140, 420, 140 );

  //*** Copy source to 1st dest. with transp. true
  gc->TranspON();
  gc->MoveRect( 160, 160, 40, 160, 240, 160 );

  //*** Copy source to 2nd dest. with transp. false
  gc->TranspOFF();
  gc->MoveRect( 160, 160, 40, 160, 440, 160 );
}



//----------------------------------------------------------------------------
//  This is an example image to test the AddTextSpace() function.
//  This code produces an image that has a text in which space
//  between the characters is increasing/decreasing
//----------------------------------------------------------------------------
void Test_AddTextSpace( TGC* gc )
{
  INT16 x, y, i;
  char  *s;
  INT32 os = gc->GetTextSpace();

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the active color
  gc->Transparent( TStdColor::white );

  gc->SelectFont( "OPIX9" );

  s = "Note increasing spaces between characters.";

  for( i = -8, y = 0; i < 20; ++i )
  {
    gc->AddTextSpace( i );
    x = 320 - gc->GetWidth(s)/2;
    y += gc->CharHigh();
    gc->DrawText( x, y, s );
  }

  gc->AddTextSpace( os );
}


//----------------------------------------------------------------------------
//  This is an example image to test the GetTextSpace() function.
//  This code produces an image that has a text in which space
//  between the characters is increasing
//----------------------------------------------------------------------------
void Test_GetTextSpace( TGC* gc )
{
  INT16 x, y, i;
  char  *s;
  char  s1[32];
  INT32 os = gc->GetTextSpace();

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the active color
  gc->Transparent( TStdColor::white );

  gc->SelectFont( "OPIX9" );

  s = "Note increasing spaces";

  for( i = 0, y = 0; i < 20; ++i )
  {
    gc->AddTextSpace(i);
    x = 320 - gc->GetWidth(s)/2;
    y += gc->CharHigh();
    gc->DrawString( x, y, s );
    sprintf( s1, "%d", gc->GetTextSpace() );
    gc->DrawString( x - 40, y, s1 );
  }

  gc->AddTextSpace( os );
}


//----------------------------------------------------------------------------
//  This is an example image to test the AddWordSpace() function.
//  This code produces an image that has a text in which space
//  between the words is increasing
//----------------------------------------------------------------------------
void Test_AddWordSpace( TGC* gc )
{
  INT32 x, y, i;
  char  *s;
  INT32 os = gc->GetWordSpace();

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the active color
  gc->Transparent( TStdColor::white );

  gc->SelectFont( "OPIX9" );

  s = "Note increasing spaces between words.";

  for( i = 0, y = 0; i < 20; ++i )
  {
    gc->AddWordSpace(i);
    x = 320 - gc->GetWidth(s)/2;
    y += gc->CharHigh();
    gc->DrawText( x, y, s );
  }

  gc->AddWordSpace( os );
}

//----------------------------------------------------------------------------
//  This is an example image to test the GetWordSpace() function
//  this code produces an image that has a text in which space
//  between the words is increasing
//----------------------------------------------------------------------------
void Test_GetWordSpace( TGC* gc )
{
  INT16 x, y, i;
  char  *s;
  char  s1[32];
  INT32 os = gc->GetWordSpace();

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the active color
  gc->Transparent( TStdColor::white );

  gc->SelectFont( "OPIX9" );

  s = "Note increasing spaces between words.";

  for( i = 0, y = 0; i < 20; ++i )
  {
    gc->AddWordSpace(i);
    x = 320 - gc->GetWidth(s)/2;
    y += gc->CharHigh();
    gc->DrawText( x, y, s );
    sprintf( s1, "%d", gc->GetWordSpace() );
    gc->DrawString( x - 40, y, s1 );
  }

  gc->AddWordSpace( os );
}


//----------------------------------------------------------------------------
//  This is an example image to test the AddMoreSpace() function.
//  This code produces an image that has a text in which space
//  between some words words is increasing
//----------------------------------------------------------------------------
void Test_AddMoreSpace( TGC* gc )
{
  INT16 x, y, i;
  char  *s;
  INT32 os = gc->GetMoreSpace();

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the active color
  gc->Transparent( TStdColor::white );

  gc->SelectFont( "OPIX9" );

  s = "Note increasing spaces between words.";

  for( i = 0, y = 0; i < 20; ++i )
  {
    gc->AddMoreSpace(i);
    x = 320 - gc->GetWidth(s)/2;
    y += gc->CharHigh();
    gc->DrawText( x, y, s );
  }

  gc->AddMoreSpace( os );
}


//----------------------------------------------------------------------------
//  This is an example image to test the GetMoreSpace() function.
//  This code produces an image that has a text in which space
//  between some words words is increasing
//----------------------------------------------------------------------------
void Test_GetMoreSpace( TGC* gc )
{
  INT16 x, y, i;
  char  *s;
  char  s1[32];
  INT32 os = gc->GetMoreSpace();

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** Set the active color
  gc->Transparent( TStdColor::white );

  gc->SelectFont( "OPIX9" );

  s = "Note increasing spaces between words.";

  for( i = 0, y = 0; i < 20; ++i )
  {
    gc->AddMoreSpace(i);
    x = 320 - gc->GetWidth(s)/2;
    y += gc->CharHigh();
    gc->DrawText( x, y, s );
    sprintf( s1, "%d", gc->GetMoreSpace() );
    gc->DrawString( x - 40, y, s1 );
  }

  gc->AddMoreSpace( os );
}


//----------------------------------------------------------------------------
//  This is an example image to test the GetFontMax() function.
//  This code produces an image that has a numeric text which
//  represents the maximum number of fonts available
//----------------------------------------------------------------------------
void Test_GetFontMax( TGC* gc )
{
  INT16 x, y, i;
  char  s1[32];

  //*** Setup look-up-table for colors white and black and red green blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen
  gc->BlackFillVuport();

  //*** set the active color
  gc->Transparent( TStdColor::white );

  gc->SelectFont( "BIG30" );

  sprintf( s1, "%d", gc->GetFontMax() );
  gc->DrawString( 320, 240, s1 );
}


//----------------------------------------------------------------------------
//  This is an example image to test the GetRect() function.
//  This code produces an image that has a shape surrounded by
//  copies of itself
//----------------------------------------------------------------------------
void Test_GetRect( TGC* gc )
{
  INT32   w, h, pitch32, i;
  INT32   x, y;
  INT32   bitwidth;
  UINT32 *buf;

  w = 80;
  h = 60;
  x = 280;
  y = 210;

  bitwidth = gc->Get_psize() * w;
  pitch32 = bitwidth / 32;
  if( bitwidth % 32 ) pitch32++;

  buf = (UINT32*)malloc( pitch32 * h * sizeof(UINT32) );
 	if( buf == NULL )	return;

  //*** Setup look-up-table for colors white and black and colors red, green
  //*** and blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen with black.
  gc->BlackFillVuport();

  gc->Transparent( TStdColor::red );
  gc->FillRect( w-2, h-2, x+1, y+1 );
  gc->Opaque( TStdColor::green, TStdColor::blue );
  gc->SelectPatn( "CHECKER8" );
  gc->PatnFrameOval( w-2, h-2, x+1, y+1, 20, 15 );

  gc->GetRect( w, h, x, y, buf, pitch32 );

  for( i = 25, x = 0, y = -150<<16; i > 0; --i )
  {
    x += y >> 2;
    y -= x >> 2;
    gc->PutRect( buf, pitch32, w, h, (x>>16)+280, (y>>16)+210 );
  }

  free( buf );
}


//----------------------------------------------------------------------------
//  This is an example image to test the PutPixel() function.
//  This code produces an image that has 4 boxes with flipped
//  text inside
//----------------------------------------------------------------------------
void Test_PutPixel( TGC* gc )
{
  char *s[] =
  {
    "Flip all", "the pixels", "in the box."
  };
  INT32 i, j, w, x1, y1, x2, y2;
  TColor val;

  //*** Setup look-up-table for colors white and black and colors red, green
  //*** and blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen with black.
  gc->BlackFillVuport();

  gc->SelectFont( "OPIX9" );

  //*** Draw picture to be flipped
  gc->Transparent( TStdColor::white );
  y1 = 98;
  for( i = 0; i <= 2; ++i, y1 += gc->CharHigh() )
    gc->DrawString(126, y1, s[i]);
  gc->Opaque( TStdColor::blue, TStdColor::green );
  gc->SelectPatn( "CHECKER8" );
  w = 114;
  x1 = 114;
  y1 = 53;
  gc->PatnFrameRect( w, w, x1, y1, 5, 14 );

  //*** Now use PutPixel function to flip pixels
  x2 = x1 + 320;
  y2 = y1 +240;
  for( i = 0; i <= 113; ++i )
  {
    for( j = 0; j<= 113; ++j )
    {
      val = gc->GetPixel( x1+i, y1+j );
      gc->PutPixel( val, x2+w-j, y1+i   );
      gc->PutPixel( val, x2+w-i, y2+w-j );
      gc->PutPixel( val, x1+w-j, y2+w-i );
    }
  }
}



//----------------------------------------------------------------------------
//  This is an example image to test the InstallPatn() function.
//  This code produces an image that has a rectangle filled with a
//  custom 'X' pattern.
//----------------------------------------------------------------------------
void Test_InstallPatn( TGC* gc )
{
  // The original bitmap of the pattern
  /*
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,1,0,0,0,0,0,0,0,0,1,0,0,0,
  0,0,1,1,1,0,0,0,0,0,0,1,1,1,0,0,
  0,1,1,1,1,1,0,0,0,0,1,1,1,1,1,0,
  0,0,1,1,1,1,1,0,0,1,1,1,1,1,0,0,
  0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,
  0,0,0,0,1,1,1,1,1,1,1,1,0,0,0,0,
  0,0,0,0,0,1,1,1,1,1,1,0,0,0,0,0,
  0,0,0,0,0,1,1,1,1,1,1,0,0,0,0,0,
  0,0,0,0,1,1,1,1,1,1,1,1,0,0,0,0,
  0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,
  0,0,1,1,1,1,1,0,0,1,1,1,1,1,0,0,
  0,1,1,1,1,1,0,0,0,0,1,1,1,1,1,0,
  0,0,1,1,1,0,0,0,0,0,0,1,1,1,0,0,
  0,0,0,1,0,0,0,0,0,0,0,0,1,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
  */

  //*** Change the bitmap to an array of INT16's with the left most bit in the
  //*** bitmap is the MSB in the INT16
  INT16 mypatn[] =
  {
     0x0000,
     0x1008,
     0x381c,
     0x7c3e,
     0x3e7c,
     0x1ff8,
     0x0ff0,
     0x07e0,
     0x07e0,
     0x0ff0,
     0x1ff8,
     0x3e7c,
     0x7c3e,
     0x381c,
     0x1008,
     0x0000
  };

  //*** Setup look-up-table for colors white and black and colors red, green
  //*** and blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen with black.
  gc->BlackFillVuport();

  //*** Install (and select) our custom pattern
  //*** Could also select with gc->SelectPatn( "User" );
  gc->InstallPatn ( mypatn );
  gc->Opaque      ( TStdColor::blue, TStdColor::red);
  gc->PatnFillRect( 448, 288, 96, 96);
}


//----------------------------------------------------------------------------
//  This is an example image to test the RemovePatn() function.
//  This code produces an image that has a solid and a pattern
//  filled rectangles
//----------------------------------------------------------------------------
void Test_RemovePatn( TGC* gc )
{
  // The origina l bitmap of the pattern
  /*
     0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
     0,0,0,1,0,0,0,0,0,0,0,0,1,0,0,0,
     0,0,1,1,1,0,0,0,0,0,0,1,1,1,0,0,
     0,1,1,1,1,1,0,0,0,0,1,1,1,1,1,0,
     0,0,1,1,1,1,1,0,0,1,1,1,1,1,0,0,
     0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,
     0,0,0,0,1,1,1,1,1,1,1,1,0,0,0,0,
     0,0,0,0,0,1,1,1,1,1,1,0,0,0,0,0,
     0,0,0,0,0,1,1,1,1,1,1,0,0,0,0,0,
     0,0,0,0,1,1,1,1,1,1,1,1,0,0,0,0,
     0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,
     0,0,1,1,1,1,1,0,0,1,1,1,1,1,0,0,
     0,1,1,1,1,1,0,0,0,0,1,1,1,1,1,0,
     0,0,1,1,1,0,0,0,0,0,0,1,1,1,0,0,
     0,0,0,1,0,0,0,0,0,0,0,0,1,0,0,0,
     0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
   */

  //*** Change the bitmap to array of shorts with the left most bit in the
  //*** bitmap is the MSB in the short
  INT16 mypatn[] =
  {
     0x0000,
     0x1008,
     0x381c,
     0x7c3e,
     0x3e7c,
     0x1ff8,
     0x0ff0,
     0x07e0,
     0x07e0,
     0x0ff0,
     0x1ff8,
     0x3e7c,
     0x7c3e,
     0x381c,
     0x1008,
     0x0000
  };

  //*** Setup look-up-table for colors white and black and colors red, green
  //*** and blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen with black.
  gc->BlackFillVuport();

  //*** Assign user index = USER_PATN
  gc->InstallPatn ( mypatn );
  gc->Opaque      ( TStdColor::blue, TStdColor::red );
  gc->PatnFillRect( 100, 100, 10, 10 );

  //*** Fill in the bitmap with 0s, so color0 is the
  //*** one which will be displayed
  gc->RemovePatn();
  gc->PatnFillRect( 100, 100, 150, 150 );
}



//----------------------------------------------------------------------------
//  This is an example image to test the GetPatnMax() function.
//  This code produces an image that has different pattern
//  filled rectangles
//----------------------------------------------------------------------------
void Test_GetPatnMax( TGC* gc )
{
  INT32 w, h, x, y, dx, dy, pmax, patn;

  //*** Setup look-up-table for colors white and black and colors reg, green
  //    and blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen with black.
  gc->BlackFillVuport();

  gc->Set_color1( TStdColor::cyan  );
  gc->Set_color0( TStdColor::white );

  //*** Display all available patterns
  pmax = gc->GetPatnMax();
  x  = y = 0;
  w  = 84;
  h  = 68;
  dx = 96;
  dy = 80;
  gc->SelectPatn( 0 );
  for( patn = 0; patn < pmax; gc->SelectPatn( ++patn ) )
  {
    gc->PatnFillRect( w, h, x, y );
    if( (x += dx) > 640 - w )
    {
      x = 0;
      y += dy;
    }
  }
}


//----------------------------------------------------------------------------
//  This is an example image to test the GetPixel() function.
//  This code produces an image that has a solid filled rectangle
//  with flipped text inside
//----------------------------------------------------------------------------
void Test_GetPixel( TGC* gc )
{
  INT32 xs, ys, xd, yd;
  char s[] = "topsy turvy";

  //*** Setup look-up-table for colors white and black and colors red, green
  //*** and blue
  gc->SetLut( BW | HUERGB );

  //*** Clear the screen with black.
  gc->BlackFillVuport();

  gc->Transparent( TStdColor::red );
  gc->FillRect   ( 150, 50, 0, 0 );

  gc->Transparent( TStdColor::white );
  gc->SelectFont ( "OPIX9" );
  gc->DrawString ( 0, gc->GetAscent(), s );

  //*** Flip and mirror original text
  for( ys = 0, yd = 29; ys <= 19; ++ys, --yd )
    for( xs = 0, xd = 89; xs <= 89; ++xs, --xd )
      gc->PutPixel( gc->GetPixel(xs,ys), xd, yd );
}


void (*(test_samples[MAX_TEST]))( TGC* ) =
{
	Test_CharWideMax,
	Test_CharHigh,
	Test_CharWideImage,
	Test_GetFirstCh,
	Test_GetLastCh,
	Test_CurtainFillRect,
	Test_DrawChar,
	Test_DrawLine,
	Test_DrawOval,
  Test_DrawPoint,
  Test_DrawPolyLine,
  Test_DrawRect,
  Test_DrawTriangle,
  Test_DrawCenterMark,
  Test_DrawLimits,
  Test_DrawCross,
  Test_DrawTBox,
  Test_DrawFormat,
  Test_DrawGrid,
  Test_DrawGrill,
  Test_DrawString,
  Test_DrawText,
  Test_GetWidth,
  Test_CharFillRect,
  Test_FillOval,
  Test_FillRect,
  Test_FillTriangle,
  Test_FillPolygon,
  Test_GetAscent,
  Test_GetDescent,
  Test_GetLeading,
  Test_MoveRect,
	Test_Opaque,
  Test_PatnFillRect,
	Test_PatnFillTriangle,
  Test_PatnFillOval,
	Test_PatnFrameOval,
  Test_FrameOval,
  Test_FrameRect,
  Test_PatnFrameRect,
  Test_SelectColor,
  Test_SetColorNamed,
  Test_SelectFont,
  Test_SelectFontNamed,
  Test_SelectPatn,
  Test_SelectPatnNamed,
  Test_PokeColor,
  Test_PokeBlackColors,
  Test_SetLevelGray,
  Test_SetLevelGrayPercent,
  Test_SetLevelRGB,
  Test_TransformDX,
  Test_TransformDY,
  Test_SquareDX,
  Test_SquareDY,
  Test_TransformX,
  Test_TransformY,
  Test_Transparent,
  Test_GroundFillVuport,
  Test_ColorFillVuport,
  Test_BlackFillVuport,
  Test_Boxes,
  Test_TranspOFF,
  Test_AddTextSpace,
  Test_GetTextSpace,
  Test_AddWordSpace,
  Test_GetWordSpace,
  Test_AddMoreSpace,
  Test_GetMoreSpace,
  Test_GetFontMax,
  Test_GetRect,
  Test_PutPixel,
  Test_InstallPatn,
  Test_RemovePatn,
  Test_GetPatnMax,
  Test_GetPixel
};

